/* ****************************************************************************
 * Copyright: 2017-2025 RAYLASE GmbH
 * This source code is the proprietary confidential property of RAYLASE GmbH.
 * Reproduction, publication, or any form of distribution to
 * any party other than the licensee is strictly prohibited.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#ifndef CORRECTION_FILE_H_
#define CORRECTION_FILE_H_

#include "rlTypes.h"

#ifdef _WIN32
	#ifdef RLCF_COMPILING_DLL
		#define RLCF_API __declspec(dllexport)
	#else
		#define RLCF_API __declspec(dllimport)
	#endif
#elif defined(__GNUC__) || defined(__clang__)
	#define RLCF_API __attribute__((visibility("default")))
#else
	#define RLCF_API
#endif
#ifdef __cplusplus
	#define RLCF_C_API extern "C" RLCF_API
#else
	#define RLCF_C_API RLCF_API
#endif

typedef int64_t rlCFHandle;
typedef int64_t rlInverseCFHandle;

RLCF_C_API rlCFHandle rlCFOpenRead(const char* fileName);
RLCF_C_API rlCFHandle rlCFOpenWrite(const char* fileName);
RLCF_C_API rlCFHandle rlCFCreate(
    const char* fileName, const rlOpticalParameters* param, const char* const* correctionTables, const uint32_t* correctionLen, uint32_t tableCount);
RLCF_C_API rlCFHandle rlCFCreateWithExtraInfo(const char* fileName, const rlOpticalParameters* param, const char* const* correctionTables,
    const uint32_t* correctionLen, uint32_t tableCount, const char* extraInfo);
RLCF_C_API rlCFHandle rlCFCreateWithScannerInfo(const char* fileName, const rlOpticalParameters* param, const char* const* correctionTables,
    const uint32_t* correctionLen, uint32_t tableCount, const rlScannerInfo* scannerInfo);
RLCF_C_API rlCFHandle rlCFCreateWithExtraInfoScannerInfo(const char* fileName, const rlOpticalParameters* param, const char* const* correctionTables,
    const uint32_t* correctionLen, uint32_t tableCount, const char* extraInfo, const rlScannerInfo* scannerInfo);

RLCF_C_API rlResult rlCFClose(rlCFHandle fileHandle);
RLCF_C_API rlResult rlCFIsInverseCorrection(rlCFHandle fileHandle, bool* out);
RLCF_C_API rlResult rlCFIs4D(rlCFHandle fileHandle, bool* out);
RLCF_C_API rlResult rlCFGetOpticalParameters(rlCFHandle fileHandle, rlOpticalParameters* out);
RLCF_C_API rlResult rlCFTableReadByte(rlCFHandle fileHandle, char* buffer, uint32_t bufferLen, uint32_t offset, uint32_t count, uint32_t table);
RLCF_C_API rlResult rlCFTableReadUInt64(rlCFHandle fileHandle, uint64_t* buffer, uint32_t bufferLen, uint32_t offset, uint32_t count, uint32_t table);
RLCF_C_API rlResult rlCFGetTableLen(rlCFHandle fileHandle, uint32_t* outLen, uint32_t table);
RLCF_C_API rlResult rlCFGetTableCount(rlCFHandle fileHandle, uint32_t* outCount);
RLCF_C_API rlResult rlCFHasExtraInfo(rlCFHandle fileHandle, bool* out);
RLCF_C_API rlResult rlCFGetExtraInfo(rlCFHandle fileHandle, char* buffer, uint32_t bufferSize, uint32_t* stringLength);
RLCF_C_API rlResult rlCFSetExtraInfo(rlCFHandle fileHandle, const char* extraInfo);
RLCF_C_API rlResult rlCFAppendExtraInfo(rlCFHandle fileHandle, const char* extraInfo);
RLCF_C_API rlResult rlCFHasScannerInfo(rlCFHandle fileHandle, bool* out);
RLCF_C_API rlResult rlCFGetScannerInfo(rlCFHandle fileHandle, rlScannerInfo* scannerInfo);
RLCF_C_API rlResult rlCFHasMetadata(rlCFHandle fileHandle, bool* out);
RLCF_C_API rlResult rlCFGetMetadata(rlCFHandle fileHandle, rlCFMetadataItem* map, uint32_t* outCount, uint32_t maxLen);
RLCF_C_API rlResult rlCFSetMetadata(rlCFHandle fileHandle, rlCFMetadataItem* map, uint32_t mapCount);
RLCF_C_API rlResult rlCFGetDimensions(rlCFHandle fileHandle, rlPoint4DInt* outDim);

RLCF_C_API rlResult rlCFToScanner3D(rlCFHandle fileHandle, const rlPoint3D* field, rlPoint3DInt* scannerOut);
RLCF_C_API rlResult rlCFToScanner4D(rlCFHandle fileHandle, const rlPoint4D* field, rlPoint4DInt* scannerOut);

RLCF_C_API rlInverseCFHandle rlCFGetInverse(rlCFHandle fileHandle);
RLCF_C_API rlInverseCFHandle rlInverseCFLoad(const char* fileName);
RLCF_C_API rlResult rlInverseCFClose(rlInverseCFHandle fileHandle);

RLCF_C_API rlResult rlInverseCFGetDimensions(rlInverseCFHandle fileHandle, rlPoint4DInt* outDim);
RLCF_C_API rlResult rlInverseCFGetOpticalParameters(rlInverseCFHandle fileHandle, rlOpticalParameters* out);
RLCF_C_API rlResult rlInverseCFIsInverseOf(rlInverseCFHandle fileHandle, rlCFHandle correctionFile, bool* isInverseOut);
RLCF_C_API rlResult rlInverseCFIsInverseOfOpticalParameters(rlInverseCFHandle fileHandle, const rlOpticalParameters* param, bool* isInverseOut);
RLCF_C_API rlResult rlInverseCFIsInverseOfCorrectionFile(rlInverseCFHandle fileHandle, const char* fileName, bool* isInverseOut);

RLCF_C_API rlResult rlInverseCFStore(rlInverseCFHandle fileHandle, const char* fileName);

RLCF_C_API rlResult rlInverseCFFromScanner3D(rlInverseCFHandle fileHandle, const rlPoint3DInt* scanner, rlPoint3D* fieldOut);
RLCF_C_API rlResult rlInverseCFFromScanner4D(rlInverseCFHandle fileHandle, const rlPoint4DInt* scanner, rlPoint4D* fieldOut);
RLCF_C_API rlResult rlInverseCFFromScanner4DDbl(rlInverseCFHandle fileHandle, const rlPoint4D* scanner, rlPoint4D* fieldOut);

RLCF_C_API void rlCFGetLastError(char* buffer, int32_t bufferLen);
RLCF_C_API int32_t rlCFGetLastErrorLen();

#endif // CORRECTION_FILE_H_
